import numpy as np
import matplotlib.pyplot as plt

# -----------------------------
# PHYLLOTAXIS NETWORK SETUP
# -----------------------------
num_nodes = 10       # total nodes in the network
num_strands = 8      # A-H
slots_per_strand = 4
lattice_size = (num_strands, slots_per_strand)

# Frequency mapping (example bands in Hz)
frequency_bands = np.array([100e6, 300e6, 600e6, 900e6,
                            1.2e9, 1.6e9, 2.0e9, 2.4e9])

# Network node positions using Phyllotaxis spiral
def phyllotaxis_positions(n, c=1.0):
    """Generate 2D positions using phyllotaxis formula"""
    golden_angle = np.pi * (3 - np.sqrt(5))
    positions = []
    for i in range(n):
        r = c * np.sqrt(i)
        theta = i * golden_angle
        x = r * np.cos(theta)
        y = r * np.sin(theta)
        positions.append((x, y))
    return np.array(positions)

node_positions = phyllotaxis_positions(num_nodes)

# Initialize node lattices randomly
node_lattices = [np.random.rand(*lattice_size) for _ in range(num_nodes)]

# Simulated environment signal for a given band
def environment_signal(freq, t):
    """Simulate environmental analog field as sum of sine waves"""
    return np.sin(2*np.pi*freq*t) + 0.1*np.sin(2*np.pi*freq*1.03*t)

# -----------------------------
# NETWORK LATTICE RIDE FUNCTION
# -----------------------------
def ride_network_lattice(node_lattices, freqs, t):
    """
    Each node rides environment, then lattices are blended across network.
    """
    composite = np.zeros_like(t)
    for node_idx, lattice in enumerate(node_lattices):
        # Micro-perturbation per node
        for s, freq in enumerate(freqs):
            for slot in range(lattice.shape[1]):
                # Compute perturbation weighted by node position (distance to center)
                weight = np.exp(-np.linalg.norm(node_positions[node_idx]))
                delta_phase = lattice[s, slot] * 0.1 * weight
                composite += np.sin(2*np.pi*freq*t + delta_phase)
    return composite

# -----------------------------
# SIMULATION
# -----------------------------
t = np.linspace(0, 1e-6, 5000)  # 1 microsecond sample
composite_signal = ride_network_lattice(node_lattices, frequency_bands, t)

# -----------------------------
# VISUALIZATION
# -----------------------------
plt.figure(figsize=(10,4))
plt.plot(t*1e6, composite_signal)
plt.title("Multi-Node HDGL Lattice Riding Environmental Superposition")
plt.xlabel("Time (µs)")
plt.ylabel("Amplitude")
plt.grid(True)
plt.show()

# Visualize node layout (phyllotaxis)
plt.figure(figsize=(5,5))
plt.scatter(node_positions[:,0], node_positions[:,1], c='orange', s=100)
plt.title("Phyllotaxis Node Layout")
plt.axis('equal')
plt.grid(True)
plt.show()
